<?php
if (!defined('ABSPATH')) { exit; }

class TRV_Api_Client {
  private $opts;
  public function __construct($opts) { $this->opts = $opts; }
  private function base_url() { return untrailingslashit($this->opts['api_base']); }
  private function slug() { return $this->opts['company_slug']; }

  private function headers($etag = null) {
    $h = ['Accept'=>'application/json','User-Agent'=>'Taxireview-WP-Embed/'.TRV_WP_EMBED_VERSION,'X-TRV-Site'=>home_url()];
    if (!empty($this->opts['api_key'])) $h['Authorization'] = 'Bearer '.$this->opts['api_key'];
    if ($etag) $h['If-None-Match'] = $etag;
    return $h;
  }

  private function cache_key($suffix) {
    return 'trv_' . md5($this->base_url().'|'.$this->slug().'|'.$suffix);
  }

  public function purge_cache() {
    foreach (['summary','etag_summary'] as $k) { delete_transient($this->cache_key($k)); delete_option($this->cache_key($k)); }
  }

  public function get_summary($force=false) {
    $slug = $this->slug();
    if (empty($slug)) return new WP_Error('trv_missing_slug', __('Vul eerst de bedrijf slug in bij de plugin-instellingen.', 'taxireview-wp-embed'));
    $ck = $this->cache_key('summary');
    if (!$force) { $c = get_transient($ck); if ($c) return $c; }

    $url = $this->base_url().'/wp-json/taxireview/v1/company/'.rawurlencode($slug).'/summary';
    $etag_key = $this->cache_key('etag_summary');
    $etag = get_option($etag_key);

    $resp = wp_remote_get($url, ['timeout'=>10,'headers'=>$this->headers($etag)]);
    if (is_wp_error($resp)) { $stale=get_transient($ck); return $stale ?: $resp; }

    $code = wp_remote_retrieve_response_code($resp);
    if ($code === 304) { $c=get_transient($ck); if ($c) return $c; }

    $data = json_decode(wp_remote_retrieve_body($resp), true);
    if ($code >= 400 || !is_array($data)) { $stale=get_transient($ck); return $stale ?: new WP_Error('trv_api_error', sprintf(__('API fout (%d).','taxireview-wp-embed'), $code)); }

    $new_etag = wp_remote_retrieve_header($resp, 'etag');
    if ($new_etag) update_option($etag_key, $new_etag, false);

    set_transient($ck, $data, intval($this->opts['cache_ttl_summary']));
    return $data;
  }

  public function get_reviews($args=[], $force=false) {
    $slug = $this->slug();
    if (empty($slug)) return new WP_Error('trv_missing_slug', __('Vul eerst de bedrijf slug in bij de plugin-instellingen.', 'taxireview-wp-embed'));

    $defaults = ['limit'=>6,'page'=>1,'min_rating'=>1,'verified'=>1];
    $args = wp_parse_args($args, $defaults);

    $suffix = 'reviews_'.md5(wp_json_encode($args));
    $ck = $this->cache_key($suffix);
    if (!$force) { $c=get_transient($ck); if ($c) return $c; }

    $url = add_query_arg([
      'limit'=>intval($args['limit']),
      'page'=>intval($args['page']),
      'min_rating'=>intval($args['min_rating']),
      'verified'=>intval($args['verified']),
    ], $this->base_url().'/wp-json/taxireview/v1/company/'.rawurlencode($slug).'/reviews');

    $etag_key = $this->cache_key('etag_'.$suffix);
    $etag = get_option($etag_key);

    $resp = wp_remote_get($url, ['timeout'=>10,'headers'=>$this->headers($etag)]);
    if (is_wp_error($resp)) { $stale=get_transient($ck); return $stale ?: $resp; }

    $code = wp_remote_retrieve_response_code($resp);
    if ($code === 304) { $c=get_transient($ck); if ($c) return $c; }

    $data = json_decode(wp_remote_retrieve_body($resp), true);
    if ($code >= 400 || !is_array($data)) { $stale=get_transient($ck); return $stale ?: new WP_Error('trv_api_error', sprintf(__('API fout (%d).','taxireview-wp-embed'), $code)); }

    $new_etag = wp_remote_retrieve_header($resp, 'etag');
    if ($new_etag) update_option($etag_key, $new_etag, false);

    set_transient($ck, $data, intval($this->opts['cache_ttl_reviews']));
    return $data;
  }

  public function build_url($base, $ref) {
    return add_query_arg([
      'utm_source'=>$this->opts['utm_source'],
      'utm_medium'=>$this->opts['utm_medium'],
      'utm_campaign'=>$this->opts['utm_campaign'],
      'utm_content'=>$ref,
      'trv_site'=>parse_url(home_url(), PHP_URL_HOST),
    ], $base);
  }
}
