<?php
if (!defined('ABSPATH')) { exit; }

class TRV_Settings {
  const OPT_KEY = 'trv_wp_embed_options';

  public static function defaults() {
    return [
      'api_base' => 'https://taxireview.nl',
      'company_slug' => '',
      'api_key' => '',
      'cache_ttl_summary' => 15 * MINUTE_IN_SECONDS,
      'cache_ttl_reviews' => 60 * MINUTE_IN_SECONDS,
      'utm_source' => 'wordpress',
      'utm_medium' => 'plugin',
      'utm_campaign' => 'reviews',
      'theme' => 'warm',
      'featured_review' => 1,
      'cta_text' => __('Schrijf een review', 'taxireview-wp-embed'),
      'view_all_text' => __('Bekijk alle reviews', 'taxireview-wp-embed'),
      'cta_style' => 'button',      // button|link
      'cta_position' => 'bottom',   // top|bottom|both|sticky (sticky only on wall)
    ];
  }

  public static function get() {
    $opts = get_option(self::OPT_KEY, []);
    return wp_parse_args($opts, self::defaults());
  }

  public static function register_menu() {
    add_options_page(
      __('Taxireview', 'taxireview-wp-embed'),
      __('Taxireview', 'taxireview-wp-embed'),
      'manage_options',
      'trv-wp-embed',
      [__CLASS__, 'render_page']
    );
  }

  public static function register_settings() {
    register_setting('trv_wp_embed', self::OPT_KEY, [
      'type' => 'array',
      'sanitize_callback' => [__CLASS__, 'sanitize'],
      'default' => self::defaults(),
    ]);

    add_settings_section('trv_main', __('Koppeling', 'taxireview-wp-embed'), function() {
      echo '<p>' . esc_html__('Koppel je Taxireview.nl profiel om reviews te tonen op je website.', 'taxireview-wp-embed') . '</p>';
    }, 'trv-wp-embed');

    foreach ([
      'api_base' => __('API basis-URL', 'taxireview-wp-embed'),
      'company_slug' => __('Bedrijf slug', 'taxireview-wp-embed'),
      'api_key' => __('API key (optioneel)', 'taxireview-wp-embed'),
    ] as $key => $label) {
      add_settings_field($key, $label, function() use ($key) {
        $opts = self::get();
        $val = $opts[$key] ?? '';
        $type = ($key === 'api_key') ? 'password' : 'text';
        $ph = ($key === 'company_slug') ? 'luchthavenexpress' : '';
        echo '<input class="regular-text" type="'.esc_attr($type).'" name="'.esc_attr(self::OPT_KEY).'['.esc_attr($key).']" value="'.esc_attr($val).'" placeholder="'.esc_attr($ph).'" />';
        if ($key === 'api_base') echo '<p class="description">'.esc_html__('Bijv. https://taxireview.nl (zonder slash aan het eind).', 'taxireview-wp-embed').'</p>';
      }, 'trv-wp-embed', 'trv_main');
    }

    
    add_settings_section('trv_appearance', __('Vormgeving', 'taxireview-wp-embed'), function() {
      echo '<p>' . esc_html__('Kies een stijl voor badge en reviews.', 'taxireview-wp-embed') . '</p>';
    }, 'trv-wp-embed');

    add_settings_field('theme', __('Stijl', 'taxireview-wp-embed'), function() {
      $o = self::get();
      $cur = $o['theme'] ?? 'warm';
      $opts = [
        'classic' => __('Klassiek (neutraal)', 'taxireview-wp-embed'),
        'warm'    => __('Warm (moderner)', 'taxireview-wp-embed'),
        'minimal' => __('Minimal (strak)', 'taxireview-wp-embed'),
      ];
      echo '<select name="'.esc_attr(self::OPT_KEY).'[theme]">';
      foreach($opts as $k=>$label){
        echo '<option value="'.esc_attr($k).'" '.selected($cur,$k,false).'>'.esc_html($label).'</option>';
      }
      echo '</select>';
      echo '<p class="description">'.esc_html__('Warm is de aanbevolen stijl.', 'taxireview-wp-embed').'</p>';
    }, 'trv-wp-embed', 'trv_appearance');

    add_settings_field('featured_review', __('Uitgelichte review', 'taxireview-wp-embed'), function() {
      $o = self::get();
      $val = !empty($o['featured_review']) ? 1 : 0;
      echo '<label><input type="checkbox" name="'.esc_attr(self::OPT_KEY).'[featured_review]" value="1" '.checked(1,$val,false).' /> ';
      echo esc_html__('Maak de eerste review in de grid/overzicht extra opvallend.', 'taxireview-wp-embed');
      echo '</label>';
    }, 'trv-wp-embed', 'trv_appearance');

add_settings_section('trv_cta', __('CTA & tracking', 'taxireview-wp-embed'), function() {
      echo '<p>' . esc_html__('De CTA "Schrijf een review" is verplicht en wordt overal getoond. Je kunt alleen tekst/stijl/positie aanpassen.', 'taxireview-wp-embed') . '</p>';
    }, 'trv-wp-embed');

    add_settings_field('cta_text', __('CTA tekst', 'taxireview-wp-embed'), function() {
      $o = self::get();
      echo '<input class="regular-text" type="text" name="'.esc_attr(self::OPT_KEY).'[cta_text]" value="'.esc_attr($o['cta_text']).'" />';
    }, 'trv-wp-embed', 'trv_cta');

    add_settings_field('view_all_text', __('"Bekijk alles" tekst', 'taxireview-wp-embed'), function() {
      $o = self::get();
      echo '<input class="regular-text" type="text" name="'.esc_attr(self::OPT_KEY).'[view_all_text]" value="'.esc_attr($o['view_all_text']).'" />';
    }, 'trv-wp-embed', 'trv_cta');

    add_settings_field('cta_style', __('CTA stijl', 'taxireview-wp-embed'), function() {
      $o = self::get(); $v = $o['cta_style'];
      echo '<select name="'.esc_attr(self::OPT_KEY).'[cta_style]">';
      foreach (['button'=>__('Knop','taxireview-wp-embed'),'link'=>__('Tekstlink','taxireview-wp-embed')] as $k=>$lab) {
        echo '<option value="'.esc_attr($k).'" '.selected($v,$k,false).'>'.esc_html($lab).'</option>';
      }
      echo '</select>';
    }, 'trv-wp-embed', 'trv_cta');

    add_settings_field('cta_position', __('CTA positie', 'taxireview-wp-embed'), function() {
      $o = self::get(); $v = $o['cta_position'];
      echo '<select name="'.esc_attr(self::OPT_KEY).'[cta_position]">';
      foreach (['top'=>__('Boven','taxireview-wp-embed'),'bottom'=>__('Onder','taxireview-wp-embed'),'both'=>__('Boven + onder','taxireview-wp-embed'),'sticky'=>__('Sticky (alleen wall)','taxireview-wp-embed')] as $k=>$lab) {
        echo '<option value="'.esc_attr($k).'" '.selected($v,$k,false).'>'.esc_html($lab).'</option>';
      }
      echo '</select>';
    }, 'trv-wp-embed', 'trv_cta');

    add_settings_field('utm', __('UTM defaults', 'taxireview-wp-embed'), function() {
      $o = self::get();
      foreach (['utm_source','utm_medium','utm_campaign'] as $k) {
        echo '<p><label>'.esc_html($k).'<br/><input class="regular-text" type="text" name="'.esc_attr(self::OPT_KEY).'['.esc_attr($k).']" value="'.esc_attr($o[$k]).'" /></label></p>';
      }
    }, 'trv-wp-embed', 'trv_cta');

    add_settings_section('trv_cache', __('Cache', 'taxireview-wp-embed'), function() {
      echo '<p>' . esc_html__('Caching maakt de widgets snel en beperkt API-verkeer.', 'taxireview-wp-embed') . '</p>';
    }, 'trv-wp-embed');

    add_settings_field('cache_ttl_summary', __('Cache TTL (summary) seconden', 'taxireview-wp-embed'), function() {
      $o = self::get();
      echo '<input type="number" min="60" step="60" name="'.esc_attr(self::OPT_KEY).'[cache_ttl_summary]" value="'.esc_attr(intval($o['cache_ttl_summary'])).'" />';
    }, 'trv-wp-embed', 'trv_cache');

    add_settings_field('cache_ttl_reviews', __('Cache TTL (reviews) seconden', 'taxireview-wp-embed'), function() {
      $o = self::get();
      echo '<input type="number" min="60" step="60" name="'.esc_attr(self::OPT_KEY).'[cache_ttl_reviews]" value="'.esc_attr(intval($o['cache_ttl_reviews'])).'" />';
    }, 'trv-wp-embed', 'trv_cache');
  }

  public static function sanitize($in) {
    $d = self::defaults();
    $o = [];
    $o['api_base'] = isset($in['api_base']) ? untrailingslashit(esc_url_raw($in['api_base'])) : $d['api_base'];
    $o['company_slug'] = isset($in['company_slug']) ? sanitize_title($in['company_slug']) : '';
    $o['api_key'] = isset($in['api_key']) ? sanitize_text_field($in['api_key']) : '';
    $o['cache_ttl_summary'] = isset($in['cache_ttl_summary']) ? max(60, intval($in['cache_ttl_summary'])) : $d['cache_ttl_summary'];
    $o['cache_ttl_reviews'] = isset($in['cache_ttl_reviews']) ? max(60, intval($in['cache_ttl_reviews'])) : $d['cache_ttl_reviews'];
    foreach (['utm_source','utm_medium','utm_campaign'] as $k) $o[$k] = isset($in[$k]) ? sanitize_text_field($in[$k]) : $d[$k];
    $o['cta_text'] = isset($in['cta_text']) ? sanitize_text_field($in['cta_text']) : $d['cta_text'];
    $o['view_all_text'] = isset($in['view_all_text']) ? sanitize_text_field($in['view_all_text']) : $d['view_all_text'];
    $o['cta_style'] = (isset($in['cta_style']) && in_array($in['cta_style'], ['button','link'], true)) ? $in['cta_style'] : $d['cta_style'];
    $o['cta_position'] = (isset($in['cta_position']) && in_array($in['cta_position'], ['top','bottom','both','sticky'], true)) ? $in['cta_position'] : $d['cta_position'];
    $allowed_themes = ['classic','warm','minimal'];
    $theme = isset($in['theme']) ? sanitize_key($in['theme']) : $d['theme'];
    $o['theme'] = in_array($theme, $allowed_themes, true) ? $theme : $d['theme'];
    $o['featured_review'] = !empty($in['featured_review']) ? 1 : 0;


    return $o;
  }

  public static function render_page() {
    if (!current_user_can('manage_options')) return;
    $opts = self::get();
    $client = new TRV_Api_Client($opts);
    $test = null;

    if (isset($_POST['trv_test']) && check_admin_referer('trv_test_nonce')) $test = $client->get_summary(true);
    if (isset($_POST['trv_purge']) && check_admin_referer('trv_purge_nonce')) $client->purge_cache();

    echo '<div class="wrap"><h1>'.esc_html__('Taxireview','taxireview-wp-embed').'</h1>';
    echo '<form method="post" action="options.php">';
    settings_fields('trv_wp_embed'); do_settings_sections('trv-wp-embed'); submit_button();
    echo '</form><hr/>';

    echo '<h2>'.esc_html__('Diagnose','taxireview-wp-embed').'</h2>';
    echo '<form method="post" style="display:flex; gap:8px; align-items:center;">';
    wp_nonce_field('trv_test_nonce');
    submit_button(__('Test verbinding','taxireview-wp-embed'), 'secondary', 'trv_test', false);
    echo '</form>';

    echo '<form method="post" style="margin-top:8px;">';
    wp_nonce_field('trv_purge_nonce');
    submit_button(__('Cache leegmaken','taxireview-wp-embed'), 'secondary', 'trv_purge', false);
    echo '</form>';

    if ($test !== null) {
      echo '<div style="margin-top:12px; padding:12px; border:1px solid #ccd0d4; background:#fff;">';
      if (is_wp_error($test)) echo '<p><strong>'.esc_html__('Fout:','taxireview-wp-embed').'</strong> '.esc_html($test->get_error_message()).'</p>';
      else { echo '<p><strong>OK:</strong> '.esc_html__('Summary geladen.','taxireview-wp-embed').'</p>';
        echo '<pre style="white-space:pre-wrap; max-width:100%;">'.esc_html(wp_json_encode($test, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE)).'</pre>';
      }
      echo '</div>';
    }

    echo '<hr/><h2>'.esc_html__('Gebruik','taxireview-wp-embed').'</h2><ul>';
    echo '<li><code>[taxireview_badge]</code></li>';
    echo '<li><code>[taxireview_reviews layout="grid" limit="6"]</code></li>';
    echo '<li><code>[taxireview_reviews layout="carousel" limit="8"]</code></li>';
    echo '<li><code>[taxireview_wall per_page="10"]</code></li>';
    echo '<li><code>[taxireview_qr size="220"]</code></li>';
    echo '</ul></div>';
  }
}
