<?php
if (!defined('ABSPATH')) { exit; }

class TRV_Render {

  public static function stars($rating, $scale=5) {
    $rating=floatval($rating); $full=floor($rating); $half=(($rating-$full)>=0.5)?1:0;
    $empty=max(0, intval($scale)-intval($full)-intval($half));
    $out='<span class="trv-stars" aria-label="'.esc_attr(sprintf(__('Waardering %.1f van %d','taxireview-wp-embed'), $rating, $scale)).'">';
    for($i=0;$i<$full;$i++) $out.='★';
    if($half) $out.='½';
    for($i=0;$i<$empty;$i++) $out.='☆';
    return $out.'</span>';
  }

  public static function cta($summary, $ref, $opts, $client) {
    $review_url = $summary['urls']['review_short'] ?? ($summary['urls']['review'] ?? '');
    $profile_url = $summary['urls']['profile'] ?? '';
    $review_href = $review_url ? esc_url($client->build_url($review_url, $ref.'_write')) : '';
    $profile_href = $profile_url ? esc_url($client->build_url($profile_url, $ref.'_profile')) : '';
    $btn_class = ($opts['cta_style']==='link') ? 'trv-link' : 'trv-button';

    $out='<div class="trv-cta">';
    if($review_href) $out.='<a class="'.esc_attr($btn_class).'" href="'.$review_href.'" rel="noopener" target="_blank">'.esc_html($opts['cta_text']).'</a>';
    if($profile_href) $out.='<a class="trv-secondary" href="'.$profile_href.'" rel="noopener" target="_blank">'.esc_html($opts['view_all_text']).'</a>';
    return $out.'</div>';
  }

  public static function badge($summary, $ref, $opts, $client) {
    wp_enqueue_style('trv-wp-embed');
    $name=$summary['name']??''; $rating=$summary['rating']['value']??null; $count=$summary['rating']['count']??null;
    $profile=$summary['urls']['profile']??'';
    $profile_href = $profile ? esc_url($client->build_url($profile, $ref.'_badge')) : '#';

    $theme = sanitize_key($opts['theme'] ?? 'warm');
    $theme_class = 'trv-theme-' . $theme;
    $out='<div class="trv-badge '."'".' .esc_attr($theme_class).'."'".'"><a class="trv-badge-inner" href="'.$profile_href.'" rel="noopener" target="_blank">';
    $out.='<div class="trv-badge-title">'.esc_html($name).'</div>';
    if($rating!==null) $out.='<div class="trv-badge-rating">'.self::stars($rating,5).' <span class="trv-rating-number">'.esc_html(number_format_i18n(floatval($rating),1)).'</span></div>';
    if($count!==null) $out.='<div class="trv-badge-count">'.esc_html(sprintf(_n('%d review','%d reviews', intval($count), 'taxireview-wp-embed'), intval($count))).'</div>';
    $out.='<div class="trv-badge-powered"><a class="trv-verified" href="'.esc_url($client->build_url('https://taxireview.nl', $ref.'_verified')).'" rel="noopener" target="_blank">'.esc_html__('Geverifieerd via Taxireview','taxireview-wp-embed').'</a></div></a>';
    return $out . self::cta($summary, $ref, $opts, $client) . '</div>';
  }

  public static function review_card($r, $client, $ref, $is_featured=false) {
    $title=$r['title']??''; $rating=$r['rating']??null; $text=$r['text']??''; $author=$r['author']['display_name']??'';
    $published=$r['published_at']??''; $ride=$r['ride_date']??''; $permalink=$r['permalink']??''; $verified=!empty($r['is_verified']);

    $cls = $is_featured ? 'trv-review trv-review--featured' : 'trv-review';
    $out='<article class="'.esc_attr($cls).'">';
    if($title) $out.='<h3 class="trv-review-title">'.esc_html($title).'</h3>';

    $meta=[];
    if($rating!==null) $meta[]=self::stars($rating,5);
    if($author) $meta[]='<span class="trv-meta">'.esc_html($author).'</span>';
    if($published) $meta[]='<span class="trv-meta">'.esc_html($published).'</span>';
    if($ride) $meta[]='<span class="trv-meta">'.esc_html(sprintf(__('Ritdatum: %s','taxireview-wp-embed'), $ride)).'</span>';
    if($meta) $out.='<div class="trv-review-meta">'.implode(' · ', $meta).'</div>';

    if($text) {
      $out.='<div class="trv-review-text trv-clamp">'.wp_kses_post(wpautop($text)).'</div>';
      if($permalink) {
        $out.='<button type="button" class="trv-readmore" data-trv-toggle>'.esc_html__('Lees meer', 'taxireview-wp-embed').'</button>';
      }
    }

    $label = $verified ? __('Geverifieerd via Taxireview','taxireview-wp-embed') : __('Review via Taxireview','taxireview-wp-embed');
    if($permalink) $out.='<div class="trv-review-footer"><a class="trv-verified" href="'.esc_url($client->build_url('https://taxireview.nl', $ref.'_verified')).'" rel="noopener" target="_blank">'.esc_html($label).'</a></div>';
    else $out.='<div class="trv-review-footer"><span class="trv-verified">'.esc_html($label).'</span></div>';

    return $out.'</article>';
  }

  public static function reviews_grid($summary, $reviews, $ref, $opts, $client) {
    wp_enqueue_style('trv-wp-embed');
    $items=$reviews['items']??[];
    $theme = sanitize_key($opts['theme'] ?? 'warm');
    $theme_class = 'trv-theme-' . $theme;
    $out='<div class="trv-widget trv-grid '."'".' .esc_attr($theme_class).'."'".'"><div class="trv-grid-inner">';
    $i=0;
    foreach($items as $r){
      $is_featured = (!empty($opts['featured_review']) && $i===0);
      $out.=self::review_card($r,$client,$ref,$is_featured);
      $i++;
    }
    return $out.'</div>'.self::cta($summary,$ref,$opts,$client).'</div>';
  }

  public static function reviews_carousel($summary, $reviews, $ref, $opts, $client) {
    wp_enqueue_style('trv-wp-embed');
    $items = $reviews['items'] ?? [];

    $theme = sanitize_key($opts['theme'] ?? 'warm');
    $theme_class = 'trv-theme-' . $theme;
    $out = '<div class="trv-widget trv-carousel '."'".' .esc_attr($theme_class).'."'".'" data-trv-carousel>';
    $out .= '<button type="button" class="trv-arrow trv-arrow-prev" aria-label="'.esc_attr__('Vorige reviews','taxireview-wp-embed').'" data-trv-prev>‹</button>';
    $out .= '<div class="trv-carousel-track" role="list" data-trv-track>';
    foreach ($items as $r) {
      $out .= '<div class="trv-carousel-slide" role="listitem">';
      $out .= self::review_card($r, $client, $ref);
      $out .= '</div>';
    }
    $out .= '</div>';
    $out .= '<button type="button" class="trv-arrow trv-arrow-next" aria-label="'.esc_attr__('Volgende reviews','taxireview-wp-embed').'" data-trv-next>›</button>';
    $out .= self::cta($summary, $ref, $opts, $client);
    $out .= '</div>';
    return $out;
  }

public static function wall($summary, $reviews, $ref, $opts, $client, $per_page) {
    wp_enqueue_style('trv-wp-embed');
    $items=$reviews['items']??[]; $page=intval($reviews['page']??1); $total=intval($reviews['total']??0);
    $pages = $per_page>0 ? max(1,(int)ceil($total/$per_page)) : 1;

    $theme = sanitize_key($opts['theme'] ?? 'warm');
    $theme_class = 'trv-theme-' . $theme;
    $out='<div class="trv-widget trv-wall '."'".' .esc_attr($theme_class).'."'".'">';
    if(in_array($opts['cta_position'], ['top','both'], true)) $out.=self::cta($summary,$ref,$opts,$client);
    if($opts['cta_position']==='sticky') $out.='<div class="trv-sticky-cta">'.self::cta($summary,$ref,$opts,$client).'</div>';

    $i=0;
    foreach($items as $r){
      $is_featured = (!empty($opts['featured_review']) && $i===0);
      $out.=self::review_card($r,$client,$ref,$is_featured);
      $i++;
    }

    if($pages>1) {
      $out.='<nav class="trv-pagination" aria-label="'.esc_attr__('Reviews paginatie','taxireview-wp-embed').'">';
      for($i=1;$i<=$pages;$i++){
        $url=add_query_arg(['trv_page'=>$i], get_permalink());
        $cls=($i===$page)?'trv-page trv-page-current':'trv-page';
        $out.='<a class="'.esc_attr($cls).'" href="'.esc_url($url).'">'.esc_html($i).'</a>';
      }
      $out.='</nav>';
    }

    if(in_array($opts['cta_position'], ['bottom','both'], true)) $out.=self::cta($summary,$ref,$opts,$client);
    return $out.'</div>';
  }

  public static function qr($summary, $ref, $opts, $client, $size=220) {
    wp_enqueue_style('trv-wp-embed');
    $review_url = $summary['urls']['review_short'] ?? '';
    $review_href = $review_url ? $client->build_url($review_url, $ref.'_qr') : '';
    $size=max(120,min(800,intval($size)));
    $qr_src = $review_href ? 'https://chart.googleapis.com/chart?cht=qr&chs='.$size.'x'.$size.'&chl='.rawurlencode($review_href) : '';

    $out='<div class="trv-widget trv-qr"><div class="trv-qr-inner">';
    if($qr_src) $out.='<img class="trv-qr-img" src="'.esc_url($qr_src).'" width="'.esc_attr($size).'" height="'.esc_attr($size).'" alt="'.esc_attr__('QR code om een review te schrijven','taxireview-wp-embed').'" loading="lazy" />';
    $out.='<div class="trv-qr-text">'.esc_html__('Scan om een review te schrijven','taxireview-wp-embed').'</div></div>';
    return $out . self::cta($summary,$ref,$opts,$client) . '</div>';
  }
}
